import Foundation


/// Cox Proportional Hazards model implementation
/// h(t|x) = h₀(t) * exp(β₁x₁ + β₂x₂ + ... + βₚxₚ)
/// Where h₀(t) is baseline hazard and βᵢ are risk factor coefficients
class CoxProportionalHazards {
    
    /// Risk factor coefficients from medical literature
    private struct CoxCoefficients {
        // Cardiovascular risk factors (Jensen 2013, Framingham Heart Study)
        static let restingHeartRateHigh = 0.012      // Per BPM above 60
        static let hrvLow = -0.015                   // Per unit of HRV below normal
        static let vo2MaxLow = -0.025                // Per ml/kg/min below age norm
        
        // Metabolic factors (Kodama 2009, BMI meta-analysis)
        static let bmiOverweight = 0.18              // BMI 25-30
        static let bmiObese = 0.65                   // BMI 30+
        static let bmiSeverelyObese = 1.20           // BMI 35+
        
        // Lifestyle factors (Doll 2004, British Doctors Study)
        static let smokingCurrent = 1.10             // Current smoker
        static let exerciseRegular = -0.30           // Regular exercise
        
        // Sleep and recovery (Walker 2017, sleep research)
        static let sleepDeprivation = 0.12           // <6 hours per night
        static let sleepExcessive = 0.08             // >9 hours per night
        
        // Physical activity (Lee 2012, activity research)
        static let sedentaryLifestyle = 0.20         // <5000 steps/day
        static let highActivity = -0.15              // >10000 steps/day
    }
    
    /// Calculate hazard ratio for given health metrics
    /// Returns multiplicative factor for baseline mortality
    func calculateHazardRatio(healthMetrics: HealthMetrics, age: Double, gender: Gender) -> Double {
        var logHazardRatio: Double = 0.0
        
        // Resting heart rate assessment
        if let heartRate = healthMetrics.restingHeartRate {
            logHazardRatio += assessHeartRateRisk(heartRate: Double(heartRate))
        }
        
        // Heart rate variability assessment  
        if let hrv = healthMetrics.hrv {
            logHazardRatio += assessHRVRisk(hrv: hrv, age: age)
        }
        
        // VO2 Max assessment
        if let vo2Max = healthMetrics.vo2Max {
            logHazardRatio += assessVO2MaxRisk(vo2Max: vo2Max, age: age, gender: gender)
        }
        
        // BMI assessment
        if let bmi = healthMetrics.bmi {
            logHazardRatio += assessBMIRisk(bmi: bmi)
        }
        
        // Sleep assessment
        if let sleepHours = healthMetrics.sleepHours {
            logHazardRatio += assessSleepRisk(sleepHours: sleepHours)
        }
        
        // Step count assessment
        if let stepCount = healthMetrics.stepCount {
            logHazardRatio += assessActivityRisk(stepCount: stepCount)
        }
        
        // Convert log hazard ratio to hazard ratio
        let hazardRatio = exp(logHazardRatio)
        
        // Bounds checking to prevent extreme values
        return max(0.1, min(10.0, hazardRatio))
    }
    
    /// Convenience method for calculating hazard ratio with baseline parameters
    func calculateBaselineHazardRatio(healthMetrics: HealthMetrics) -> Double {
        // Use default age and gender for baseline calculations
        return calculateHazardRatio(healthMetrics: healthMetrics, age: 45.0, gender: .male)
    }
    
    /// Calculate baseline hazard h₀(t) using population mortality data
    /// This represents the hazard for an individual with average risk factors
    func calculateBaselineHazard(at age: Double) -> Double {
        // Using simplified Gompertz-Makeham baseline mortality
        // These parameters are calibrated for general population
        let A = 0.0001  // Age-independent mortality rate
        let B = 0.00003 // Initial mortality rate at birth
        let C = 0.08    // Rate of mortality increase with age
        
        // Gompertz-Makeham: h₀(t) = A + B * exp(C * t)
        let baselineHazard = A + B * exp(C * age)
        
        // Convert to annual mortality rate
        return baselineHazard
    }
    
    /// Full Cox proportional hazards calculation: h(t|x) = h₀(t) * exp(β'x)
    /// Combines baseline hazard with personalized risk factors
    func calculatePersonalizedHazard(at age: Double, healthMetrics: HealthMetrics, gender: Gender) -> Double {
        let baselineHazard = calculateBaselineHazard(at: age)
        let hazardRatio = calculateHazardRatio(healthMetrics: healthMetrics, age: age, gender: gender)
        
        return baselineHazard * hazardRatio
    }
    
    /// Convert hazard rate to survival probability over time period
    func calculateSurvivalProbability(hazardRate: Double, timeYears: Double) -> Double {
        // S(t) = exp(-∫h(u)du) ≈ exp(-h*t) for constant hazard
        return exp(-hazardRate * timeYears)
    }
    
    /// Calculate life expectancy using Cox model with numerical integration
    func calculateLifeExpectancy(currentAge: Double, healthMetrics: HealthMetrics, gender: Gender, maxAge: Double = 120.0) -> Double {
        let stepSize = 0.5  // Half-year steps for integration
        var totalLifeExpectancy = 0.0
        var survivalProbability = 1.0
        
        var age = currentAge
        while age < maxAge && survivalProbability > 0.001 {
            let hazardRate = calculatePersonalizedHazard(at: age, healthMetrics: healthMetrics, gender: gender)
            let stepSurvival = calculateSurvivalProbability(hazardRate: hazardRate, timeYears: stepSize)
            
            // Add expected survival time for this age interval
            totalLifeExpectancy += survivalProbability * stepSurvival * stepSize
            
            // Update survival probability
            survivalProbability *= stepSurvival
            age += stepSize
        }
        
        return totalLifeExpectancy
    }
    
    /// Assess cardiovascular risk from resting heart rate
    private func assessHeartRateRisk(heartRate: Double) -> Double {
        // Optimal resting heart rate is 60-100 BPM
        // Risk increases above 70 BPM (Jensen 2013)
        let optimalHeartRate: Double = 60
        let riskThreshold: Double = 70
        
        guard heartRate > riskThreshold else { return 0.0 }
        
        let excessBPM = heartRate - optimalHeartRate
        return CoxCoefficients.restingHeartRateHigh * excessBPM
    }
    
    /// Assess autonomic nervous system health from HRV
    private func assessHRVRisk(hrv: Double, age: Double) -> Double {
        // HRV normally decreases with age
        // Normal HRV ranges: 20s: 30-75ms, 60s: 15-40ms
        let expectedHRV = calculateExpectedHRV(age: age)
        let hrvDeficit = max(0, expectedHRV - hrv)
        
        return CoxCoefficients.hrvLow * (hrvDeficit / expectedHRV)
    }
    
    /// Calculate expected HRV for age
    private func calculateExpectedHRV(age: Double) -> Double {
        // Age-adjusted HRV expectations (Umetani 1998)
        let baseHRV = 50.0  // Baseline for age 20
        let ageDecline = 0.4  // Decline per year
        
        return max(15.0, baseHRV - (age - 20) * ageDecline)
    }
    
    /// Assess cardiorespiratory fitness risk from VO2 Max
    private func assessVO2MaxRisk(vo2Max: Double, age: Double, gender: Gender) -> Double {
        let expectedVO2Max = calculateExpectedVO2Max(age: age, gender: gender)
        let vo2Deficit = max(0, expectedVO2Max - vo2Max)
        
        return CoxCoefficients.vo2MaxLow * (vo2Deficit / expectedVO2Max)
    }
    
    /// Calculate age and gender-adjusted expected VO2 Max
    private func calculateExpectedVO2Max(age: Double, gender: Gender) -> Double {
        // Age-adjusted VO2 Max expectations (ACSM guidelines)
        let baseFitness = gender == .male ? 45.0 : 35.0
        let ageDecline = 0.5  // ml/kg/min decline per year after 30
        let ageAdjustment = max(0, age - 30) * ageDecline
        
        return max(gender == .male ? 25.0 : 20.0, baseFitness - ageAdjustment)
    }
    
    /// Assess metabolic risk from BMI
    private func assessBMIRisk(bmi: Double) -> Double {
        switch bmi {
        case ..<18.5:
            return 0.1  // Underweight risk
        case 18.5..<25:
            return 0.0  // Normal weight
        case 25..<30:
            return CoxCoefficients.bmiOverweight
        case 30..<35:
            return CoxCoefficients.bmiObese
        default:
            return CoxCoefficients.bmiSeverelyObese
        }
    }
    
    /// Assess sleep quality risk
    private func assessSleepRisk(sleepHours: Double) -> Double {
        switch sleepHours {
        case ..<6:
            return CoxCoefficients.sleepDeprivation
        case 6...9:
            return 0.0  // Optimal sleep range
        default:
            return CoxCoefficients.sleepExcessive
        }
    }
    
    /// Assess physical activity risk from step count
    private func assessActivityRisk(stepCount: Int) -> Double {
        let steps = Double(stepCount)
        
        switch steps {
        case ..<5000:
            return CoxCoefficients.sedentaryLifestyle
        case 5000...10000:
            return 0.0  // Moderate activity
        default:
            return CoxCoefficients.highActivity  // Protective effect
        }
    }
    
    /// Calculate individual risk factor contributions
    func calculateRiskFactorContributions(healthMetrics: HealthMetrics, age: Double, gender: Gender) -> [String: Double] {
        var contributions: [String: Double] = [:]
        
        if let heartRate = healthMetrics.restingHeartRate {
            contributions["Resting Heart Rate"] = assessHeartRateRisk(heartRate: Double(heartRate))
        }
        
        if let hrv = healthMetrics.hrv {
            contributions["Heart Rate Variability"] = assessHRVRisk(hrv: hrv, age: age)
        }
        
        if let vo2Max = healthMetrics.vo2Max {
            contributions["VO2 Max"] = assessVO2MaxRisk(vo2Max: vo2Max, age: age, gender: gender)
        }
        
        if let bmi = healthMetrics.bmi {
            contributions["Body Mass Index"] = assessBMIRisk(bmi: bmi)
        }
        
        if let sleepHours = healthMetrics.sleepHours {
            contributions["Sleep Duration"] = assessSleepRisk(sleepHours: sleepHours)
        }
        
        if let stepCount = healthMetrics.stepCount {
            contributions["Physical Activity"] = assessActivityRisk(stepCount: stepCount)
        }
        
        return contributions
    }
}

/// Extensions for model validation and testing
extension CoxProportionalHazards {
    
    /// Validate that coefficients are within expected ranges from literature
    func validateCoefficients() -> Bool {
        // Check that coefficients are within reasonable ranges
        // based on published medical research
        return CoxCoefficients.smokingCurrent > 0.5 &&
               CoxCoefficients.smokingCurrent < 2.0 &&
               CoxCoefficients.exerciseRegular < 0 &&
               CoxCoefficients.exerciseRegular > -1.0
    }
    
    /// Get coefficient values for testing purposes
    func getCoefficients() -> [String: Double] {
        return [
            "restingHeartRateHigh": CoxCoefficients.restingHeartRateHigh,
            "hrvLow": CoxCoefficients.hrvLow,
            "vo2MaxLow": CoxCoefficients.vo2MaxLow,
            "bmiOverweight": CoxCoefficients.bmiOverweight,
            "bmiObese": CoxCoefficients.bmiObese,
            "smokingCurrent": CoxCoefficients.smokingCurrent,
            "exerciseRegular": CoxCoefficients.exerciseRegular,
            "sleepDeprivation": CoxCoefficients.sleepDeprivation,
            "sedentaryLifestyle": CoxCoefficients.sedentaryLifestyle
        ]
    }
    
    /// Comprehensive model validation with test scenarios
    func runValidationTests() -> [String: Any] {
        var results: [String: Any] = [:]
        
        // Test scenario 1: Healthy individual
        let healthyMetrics = HealthMetrics(
            restingHeartRate: 60,
            hrv: 45.0,
            vo2Max: 35.0,
            bmi: 22.0,
            stepCount: 8000,
            sleepHours: 7.5
        )
        
        let healthyHazard = calculateHazardRatio(healthMetrics: healthyMetrics, age: 35, gender: .female)
        let healthyLifeExpectancy = calculateLifeExpectancy(currentAge: 35, healthMetrics: healthyMetrics, gender: .female)
        
        results["healthyHazardRatio"] = healthyHazard
        results["healthyLifeExpectancy"] = healthyLifeExpectancy
        
        // Test scenario 2: High-risk individual
        let highRiskMetrics = HealthMetrics(
            restingHeartRate: 95,
            hrv: 15.0,
            vo2Max: 18.0,
            bmi: 32.0,
            stepCount: 3000,
            sleepHours: 5.0
        )
        
        let highRiskHazard = calculateHazardRatio(healthMetrics: highRiskMetrics, age: 50, gender: .male)
        let highRiskLifeExpectancy = calculateLifeExpectancy(currentAge: 50, healthMetrics: highRiskMetrics, gender: .male)
        
        results["highRiskHazardRatio"] = highRiskHazard
        results["highRiskLifeExpectancy"] = highRiskLifeExpectancy
        
        // Test baseline hazard curve
        var baselineHazards: [Double] = []
        for age in stride(from: 20, through: 80, by: 10) {
            baselineHazards.append(calculateBaselineHazard(at: Double(age)))
        }
        results["baselineHazards"] = baselineHazards
        
        // Validate coefficient ranges
        results["coefficientsValid"] = validateCoefficients()
        
        return results
    }
}